function veh = createVehicle(vehConf)
%veh = createVehicle(vehConf)
% Create vehicle data for a configuration
%
% Input arguments
% ---------------
% 'library' : char
%   The path to the library folder.
%
% Outputs
% ---------------
% veh : struct
%   The vehicle configuration. Contains all of the components' data.

arguments
    vehConf
end

veh.vehType = vehConf.vehicle_type;

veh.body = load(fullfile('library', 'vehicle_body', vehConf.vehicle_name));

% Create air drag curve
if strcmp(veh.body.resForceMethod, "analytic")
    veh.body = createAirDragCurve(veh.body);
end

% Wheels
switch veh.body.axleMethod
    case "explicit"
        numTyres =  2 .* (veh.body.axles.twinTyres == 0) + 4 .* (veh.body.axles.twinTyres == 1);
    case "concentrated"
        % veh.wh = load( fullfile('library', 'wheels', veh.body.axles.tyre(1)) );
        numTyres = 2;
end
tyreNames = veh.body.axles.tyre;
tyreNames = replace(tyreNames, {' ', '/', '.'}, '_');
for n = 1:veh.body.numAxles
    tyre = load( fullfile('library', 'wheels', tyreNames(n)) );
    tyreInertia(n) = tyre.inertia * 2;
    tyreMass(n) = tyre.mass;
    tyreRadius(n) = tyre.radius;
end
veh.wh.inertia = sum( tyreInertia(:) .* numTyres(:) ) + sum( veh.body.axles.inertia );
veh.wh.mass = sum(tyreMass(:) .* numTyres(:) );
% Assume the driven axle is the second axle
veh.wh.radius = tyreRadius(2);
% veh.wh.radius = tyreRadius(2) * 0.9708;

% Final drive
veh.fd = load(fullfile('library', 'final_drive', vehConf.finalDrive_name));
veh.fd = processTransmissionMaps(veh.fd);

% Process gearbox
veh.gb = load(fullfile('library', 'transmission', vehConf.transmission_name));
veh.gb = processTransmissionMaps(veh.gb);

pwtMass = veh.wh.mass + veh.fd.mass + veh.gb.mass;

% Power sources
switch veh.vehType
    case 'conv'
        veh.eng = load(fullfile('library', 'engine', vehConf.engine_name));
        pwtMass = pwtMass + veh.eng.mass;
    case 'bev'
        veh.em = load(fullfile('library', 'e-machine', vehConf.em_name));
        veh.batt = load(fullfile('library', 'battery', vehConf.battery_name));
        pwtMass = pwtMass + veh.em.mass + veh.batt.mass;
end
if ~strcmp(vehConf.supercap_name, "")
    veh.cap = load(fullfile('library', 'supercapacitor', vehConf.supercap_name));
    pwtMass = pwtMass + veh.cap.mass;
end
if ~strcmp(vehConf.dynCharger_name, "")
    veh.dynCh = load(fullfile('library', 'dynamic_charger', vehConf.dynCharger_name));
    pwtMass = pwtMass + veh.dynCh.mass;
end

% Masses 
% Curb mass:
%   The user defines either curbMass or chassisMass
%       - If curbMass is specified, that is exactly the curb mass of the
%       vehicle and the components weights are ignored. 
%       - If chassisMass is specified instead, the curb mass is the sum of
%       chassisMass and the components' weights.
% Load mass:
%   The user defines either maxLoad or grossMass
%       - If maxLoad is specified, that is the maximum payload. 
%       - If grossMass is specified, the maximum load weight is obtained
%       as the difference between (regulatory) gross weight and curb mass;
%       this is useful for HD applications

% Curb mass
if isfield(veh.body, 'curbMass') && isfield(veh.body, 'chassisMass')
    warning("The vehicle body defines both a curbMass and a " + ...
        "chassisMass. chassisMass will be ignored.")
elseif isfield(veh.body, 'chassisMass')
    veh.body.curbMass = veh.body.chassisMass + pwtMass;
end

% Add load mass to the vehicle body mass
if isfield(veh.body, 'maxLoad') && isfield(veh.body, 'grossMass')
    warning("The vehicle body defines both a maxLoad and a " + ...
        "grossMass. grossMass will be ignored.")
elseif isfield(veh.body, 'grossMass')
    veh.body.maxLoad = veh.body.grossMass - veh.body.curbMass;
end

end